; ------------------------------------------------------------------
; The Embedded Flat Assembler UDF
; Purpose: Provide the ability to write assembly language in AutoIt
; Author:  Ward
; ------------------------------------------------------------------

; ------------------------------------------------------------------
; Functions:
;  FasmInit
;  FasmExit
;  FasmGetVersion
;  FasmGetLastError 
;  FasmCompile
;  FasmGetBinary
;  FasmGetFuncPtr
;  FasmGetBasePtr
;  FasmAdd
;  FasmGetSource
;  FasmGetSource
;  FasmReset
; ------------------------------------------------------------------

#include-once
#include <MemoryDll.au3>

Global $_FasmDll = 0, $_FasmLastError = 0

Global Const $FASM_OK 					= 0
Global Const $FASM_WORKING				= 1
Global Const $FASM_ERROR				= 2
Global Const $FASM_INVALID_PARAMETER	= -1
Global Const $FASM_OUT_OF_MEMORY		= -2
Global Const $FASM_STACK_OVERFLOW		= -3
Global Const $FASM_SOURCE_NOT_FOUND			= -4
Global Const $FASM_UNEXPECTED_END_OF_SOURCE	= -5
Global Const $FASM_CANNOT_GENERATE_CODE		= -6
Global Const $FASM_FORMAT_LIMITATIONS_EXCEDDED	= -7
Global Const $FASM_WRITE_FAILED					= -8

Global Const $FASMERR_FILE_NOT_FOUND				= -101
Global Const $FASMERR_ERROR_READING_FILE			= -102
Global Const $FASMERR_INVALID_FILE_FORMAT			= -103
Global Const $FASMERR_INVALID_MACRO_ARGUMENTS		= -104
Global Const $FASMERR_INCOMPLETE_MACRO				= -105
Global Const $FASMERR_UNEXPECTED_CHARACTERS			= -106
Global Const $FASMERR_INVALID_ARGUMENT				= -107
Global Const $FASMERR_ILLEGAL_INSTRUCTION			= -108
Global Const $FASMERR_INVALID_OPERAND 				= -109
Global Const $FASMERR_INVALID_OPERAND_SIZE			= -110
Global Const $FASMERR_OPERAND_SIZE_NOT_SPECIFIED	= -111
Global Const $FASMERR_OPERAND_SIZES_DO_NOT_MATCH	= -112
Global Const $FASMERR_INVALID_ADDRESS_SIZE			= -113
Global Const $FASMERR_ADDRESS_SIZES_DO_NOT_AGREE	= -114
Global Const $FASMERR_PREFIX_CONFLICT 				= -115
Global Const $FASMERR_LONG_IMMEDIATE_NOT_ENCODABLE	= -116
Global Const $FASMERR_RELATIVE_JUMP_OUT_OF_RANGE	= -117
Global Const $FASMERR_INVALID_EXPRESSION			= -118
Global Const $FASMERR_INVALID_ADDRESS 				= -119
Global Const $FASMERR_INVALID_VALUE					= -120
Global Const $FASMERR_VALUE_OUT_OF_RANGE			= -121
Global Const $FASMERR_UNDEFINED_SYMBOL				= -122
Global Const $FASMERR_INVALID_USE_OF_SYMBOL			= -123
Global Const $FASMERR_NAME_TOO_LONG					= -124
Global Const $FASMERR_INVALID_NAME					= -125
Global Const $FASMERR_RESERVED_WORD_USED_AS_SYMBOL	= -126
Global Const $FASMERR_SYMBOL_ALREADY_DEFINED		= -127
Global Const $FASMERR_MISSING_END_QUOTE				= -128
Global Const $FASMERR_MISSING_END_DIRECTIVE			= -129
Global Const $FASMERR_UNEXPECTED_INSTRUCTION		= -130
Global Const $FASMERR_EXTRA_CHARACTERS_ON_LINE		= -131
Global Const $FASMERR_SECTION_NOT_ALIGNED_ENOUGH	= -132
Global Const $FASMERR_SETTING_ALREADY_SPECIFIED		= -133
Global Const $FASMERR_DATA_ALREADY_DEFINED			= -134
Global Const $FASMERR_TOO_MANY_REPEATS				= -135

Global Const $_foSTRUCT		= 0
Global Const $_foSOURCE		= 1
Global Const $_foCOMPILED	= 2
Global Const $_foSTATE		= 3
Global Const $_foPASSLMT	= 4

Global Const $FASMERRINDEX_CODE			= 0
Global Const $FASMERRINDEX_MESSAGE		= 1
Global Const $FASMERRINDEX_LINENUMBER	= 2
Global Const $FASMERRINDEX_LINE			= 3


Func FasmInit($MaxStructSize = 65536, $PassesLimit = 100)
	If $_FasmDll = 0 Then
		; The binary for fasm.dll v1.68, source by XDa from http://www.opensc.ws/, update by Ward
		Local $DllBinary = '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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary &= '93884B6EC68A72F9AE02FAE0FA2225774DD765763A46660874194D7733F212741C0630EB104D10DF65DD6EE0EB01462F24460C611FAC7B66DB18E441B650FE7D6198C1ADDBEA585B30BFBF137630AAFECBFD751BB4BE0BD710B7B91A640A5A85B7581C15011BEC130BDF017F81EAC6040700E2F9D16C66B80130F7B90CC166ABD64005242810730DC1E3C9294F12ED79BA1F6008A309280C8046AFFA187B06048CB470D966E576AC14BD213BF221DF568CC4F3ED10456838B8441770BF024FD75921892C7D3B0456C1DAAFC0B0022A3EB0507A09DD8DF8EF598B151A4AC6021A29CAB7DE33D6F0C5DEC639FA68AD091B27642DD0625308485AB4B041ADED028D53448489C802F0F6E9EEDB4062535285E504486C5A1137F8007E7EFF1A75D7CB043551F6592E6CBC05913E060C3B003BE20B6EE304A84B61652074EF3CE878B8D027FFC4BB45D77D412C9B5B58DC27CC92240822890D5AECDFC6071A66AF86C4AA23842EFF7BFA7727E0F9F7D981F9FF000F2CEAFB29CB884BFEB3F4B7F64F8A66FF2956721574963C3BFD8D7689CEC55C5C28AA123C1A12B45603CF930DDC0AB77B5BB7CB6C676BBDD1E9BD0D910DB20CD66C21714E0E0A62B9AFB54FB022AAAF687F15DFBFAE3BDBFB2CD7DE3E0FCF5EC707DC400143858DC5D54E4F73DBEE0DF4D88943FC3E45F5C700F18CE0777866AB79B0325EDD63EF540F27FA52073BE9D8D82CA4B01A25F9465C5F79816D4E3C5C33AAFE013528CB37D82074886C0720643CCBB33C095C2254274030CBB34C3B446F6BE576781788E04F17C1FE5B57106BEE66DEACEBEB160DB60A9D5FF3A5FBF775B2DCDE9AEEED1A41264D85F62D101115DDDD06F04E06EF36246AAC1109BFD9A5EF4EEB061D011804244CB296FEA6A673279D073304E70E1BFFD8A7596802769AACD7AAE2FBC357F089044BEF8FCD56125FBE1A6107C4FF0B37E489E9478601C38A263A277213770A382EB46DB07906D572A10D0683C7B7688DAD24D4C321AC8D3410FC50AB2A4CEE44ADF8886E92937FDDDBD10B51716BAB3DBD23C610293E3B7C4601689703B2F07541E2FFD6016A357456018D54160266813A1A2EF5DD5BA7269E884202D7C1C8A84203B4A96FCE0604103D666978C47C0393FD088E964EF1A34FE860A0AB42747D56A8BB76366F003C04EE530789CC53AE21B8760E46A0A7A545FFBF8F607214D907FC225F7206C642FE3BFFE0D56EBB8830544ACEF4C6430EC69E15634F5C6951B7475C74338027B93AC64D5978836CCD10B52706AD368F9DB9650602B4657175D202AA42EC3B6C03B50188C1621C87C64E68A2521C5C5B602AFF67176E05028ACCB50287518B4A0C83C1038D1CE0FF36880F970D1D8D77FF8D7BFFFD7EDDC75A16FC9E608B720823011B5C8361D61FC6B0031371B03AAA0FB29C636F6D5EEC44C6035AC58B'
			$DllBinary &= '156A6E03305E6B574C572F70FB155D4D5F4731880280884201F5607FEC5E5A462A4E29CE8D6EFE8A73098DC163FF685BAF49068B5A72D939CBCAB250A80924C01443F685A5DB770210123AE243EBE368896ADB5DF34ACFE94261F18B7A2FEF6E73B71BDA9650DE12720470EB0C4FCAFBDBAC4538CB4F8B7C240429CFCE08B3859A0EDC4668B9A3D0C687E9FA5EBB5586AB833E707BC1E5166DC9BBB6BF0A94E12FE0A8134D38C0D1E583D0002639FEAE7D824F75EDF0314B4D89C581E5FF281CB604BC310A31C5B3162D2CC26DEC8488C81417A104831268A151E37609E758A28BC6421D5D988B12DAE9955DE8EB2F14DB5B04F8EBB8C59D1C81BDDE96DEFEBF0132041EF7E5FEC338CB72F5C39580FD039A51C50596E196507E0CDAB16F2ECCED88E8C1E11609CDBECA4973D8B9209058C5498EDAEEC7121EC1ED1E8356378396DEF4AF58EA8B3A09AB2B362589FAEBC6F7236CDED0C68083EA103B08284DB6D4C1E05CD85080898313B61C72D8C28964C32408370825D79272265408C9948DB52E79D15308FAB05329F4065783C20546848E6FC452B503EB115756A0C747A945632CA638C075E30B3FF9655BFA4929D9892E894A281F885D224ACB51E8D80B0BEE51625723364676C64BC95AE38D4FFF245A2826EBC42663A5028B2EB92CC9CD8E54B6420C342FF01091681C24F05CC3F2812D9294CDAC72C6FD9502B7ABB0644E49292E36E056F999D7ECF3CD86C08DB990085F87088F8D9F7A6B4F0E2A750197D13C5DC3279F7BE8ACF6D3CF681E4A7B1945C6DC037AB0FD5F660B7CDFB355570D82026964B874E0F99688604A6423427D0BC63A567FAA8025F8A8DE7B021C40DBE4212038528193238A117F11B30C7CDAAC4F1389D7B2DA341E7A1613B9040FA5DF04DD674F2BE8B3EB8FAD048AE493CFFE1C6000EB190801EB1002EB076B210FE1102909A2793EDABD1864D524743B9F1933FAA0EBE715BE0C6AAC88D0AB63BFE766E24E2D68AC4F88501EA1DA188547D6B3DB6B24F80681A5828284D0E91ADC082426205426511D2FA2D041C83D0C6D67B08D96E1DDD5DA62D28D4383C2CDC3F6D2FE1F3730E48451FFD60F09C742C302ED44101E8759AACD8221DFDB82D22B11251A83F46CE10EDB54ED6CD524D3E4EDC3651E2D31F6E7F61F80DAC353F6110DDE4BEBD58A1D286C6804FEF56E058CC9012897520887F25234DDAF41F54D6AAA4E333AD19A8DD0EEB8EF1E10A27C6856737ECCD4850DDCADC701A404D9A5B661C71668788AC128EBB87E357A88D2AFCC0A281CD40858A146F24C0AAFDC16B677C393536EFFAF2056E016B388CCD8275BF8ED7F615130D289FD89F79C6074723C2370E781AE354D8785140D211EA968F412B4DF20CEC3B21A626B1DB02B81E307C85C20FB75E943BBC4B2221416BBFE86494638824233CE8A5461F02C36'
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary &= 'D8B66060505EEEFEC2690B2DF86F1F3072083C390F86E1B9650F107124371252D40C8A5FC37712272703F2EB77E4F6E43BE4099489D30DB316999038C5B0B7A6AE8021533B832688F0BFA1958B38893A8910AF7A17C1D71A217C17806E01BBB78ADD01334608E2F57DBF1FF054C34B0120041C661222112722F60A10314B638C712566C4B82CCB300FD08C018CF8FC0C90C6F78C7C0A94A098049C6BE7FB9D2F5FAC4601180EA8C0F33CDD54A2046F75585A63B17F0FEC59C86E1173F97CF4C486000DDB7410C520B3A38D11048AED238BACB23806F9220CC1D4283B503C14B70D751B2F155BC42EC228DE750BD4FE592866558769104C76D70D17EF1020750E37085AF61BBBEB0C070613740737B087A25B673830BFB12741B161235D3D8010753DB17B9291011083C020C16D1CEB3869525698B8B62ADD3BD1051DFF25FB6618C2DC20DEACA1C80CB069B69118E147BD6C64DEC059BF00DCC639C711291754D802DCF6D1136A0B0B0307A0244BC0A9D813A9EEB113F716A0009BE8882C452176F770BD30AC74590A713BAD7B7F2136CE1A72BF1CE35593FE713BFDC0C743FCD306F81C4EEB2439FB55AC0037F4B6B5351E385DD00BE97B04A12A8B00DCAA460C98638953B6EDF03B80149CA1E310141618C5A1D9B63B0FA866E9E601BBE618978AB6A20589C39030006E2289F8C9FF32C2133B58305A20804B090187032310A3FD8753042B031B048B0DFCB7FD2F15AE95C4EB16806309FE3B95030CACFC6F5B3B1F89020F95C008C43A4B0A880AE6ED45FE6B0B886B0B9EAD3B530C73DBEC631008ED7411B31414B114DB6087115187CF44181CAD95866F6508FDF643DE4216BDCF6D5086EE116F2B0B32305B576E1013FF200809090E6AC4C9944B12FD00ACB587190E05CCD14708BC7B8E70AD460F7AB592058B5008528D172601409018F6C26E7B03BA8C13DE1054E2FE882211A0256050C750AD9B91EEF688DD5B8F7483FB45344C30C9D60A62017EE0100A8310CC1D228B082507101E0100AD2B70461A45190B02771D10DECCCDB4E804EB5E2674519660BACD1CD44B2E082437080185764D2F34231D0F7003EB29CA5BA6005F94881B670664E1B71E9FDC0A964CE21751C4FFE37023500D990F438371096A1194C6E3B954D701ED730797731D6F682A2D0B751912261420422F1594F52A1E3C5B72DFEE4CB42D010246EBF54624F03C60066823FA72073C707703F1EF88EEE89732293C2974D985D53C21CBB20045D5EB120BDB06687323D48631CEEEB181EA9EAFA982ACF502B32650BB7835267BB3DD6BB6C78ADCF6C1011C85880EBBA88DE0583EC18CDA89331291A70D048984981A9B308168851B9C85E316A9164387006991C86C8C573064F48C4F1F26EA85E9FE2E020C767107B901B52D8D1774B20AB340A3D435220546802C6BDFB668C10F'
			$DllBinary &= '4ED4750C66AD05282446D2664B88E195635F30E96D98CE80743BA0B99C507248D38F05AC0470B57A17E1684618947297730871E891233455FE2159B23C54088B6845AEB2A73231E371BB1086DF07DD59E1D0760BB4D18E0ED404DBFDD043B217188F30D9EB00DB3B85BC90A56BF86B01041E72480EF5B17BFD71FD9C07724860FD712DF93CE42590E3B079B191B1AA5D70B550B0CE8A02AC226256C08D7142B114171CB229EA00D0D4E3B083ECF8AD663D82669BB014B16EC14D31615C4BA53C56573A2251877A0D952009C57F47E5285724F7D889FE29C6721CB743BD3F1DA07214BF65A67239C1414265AA77B03B6311AFD55EC415D6D43BC26EF75B6631C9DE7051C1741CE9FFAF69842A5954B5177593C57A01EB05BC8E7AC0A3209B2FF24689D780C3AF1E2001E43CB031B077BA554E464072401AD5AD512E6438BE28DCB838036E631DA834BBFBAAA4B5086737E7AFFB89A4A5230698D982FB2496FFD8225D10CDAD158D29029D952B5B06C06104B716B587D15875AEAB0B6A2794C07C195F269D4D16611F1729091EAA8A78ED4851AB48219FB14D816B053A3AF7994E6C3D42CD173FC027A38039C6DA00B05548EED08B0E1B8A8D9693854607D7233DE459DB6BDE1E0EE38A77AE121E951BF008593DAA8B86BBF31A8E438E49C08678FF6F9F01468F24552460E31F39C94382ECA2AD96AD55833342AC72E38C3A03382F3509FA6297117C5EEBC458BFACD7103E0C3266F171406B0639B6507527D2A4AD1DEAD2EB7117A2CF30EDA3B7888D09378A334EA06B8DACB10E6CBD9CD375D2A12AAC0DB39AED6D37431F1CAD7E10888B0551F7FD7E1C9988399A88219B6578EE2501BC5CC703CF8A01F804D71691B1875487F4104FCBF7CF4558FB7E5A88430207580375761B6B7E27A01A0C878F6B0233286E1A7B08D44409FAAACE0D276C04582C52AE63E02DF8AA3A06BCD0C48F203B1D4822AC606C761093C4C31B180CEA0BB0D59A7B03DFC320EBEE6ADDF695EBB2312C5D4B8A89B78D6D91770703568BA8A3B6966779B607149418980CD1466A79A01CD98BA231FEEE44096289D929F18D7E20AA184B278AC40323C34020C770CD72F8916D09AC33C829E482AC85AC9DC4281CC27E5A621A19A08C453245090DACB2084958A373280F7563B82F5863E5770A6FFF32D8332275761AE8B0469BF00DAC2D9254547CB991885599115B177BD6551F1D6E79AB3809A1DC5A6B7B90AEB518FAAB0BC21EE62EEC0DD36E0DCA720106641053FD41B4FBECE4D92A65ABABB73E7A565BBCC9905DD8B45BFFAB4204686E0A2F370207BBBD6D51951370EB95AA6E0DFDCE842C4F418688C2A932839C8426328FD420B66F1E39675D9258540868356D01E2C6615F8A4B9E75217EEBB830C0E75F4270B412ECA44DD8B641B993AAE40757D2773758DB4DC91066B834BC'
			$DllBinary &= 'BA0E06196C77807B0218B0AA681C147C4B38668473E8065B7CB0EC33838820801E186C92931F62AA570A926B0BF540BB910E423D652905330166386C188C3128F3122527B81A04B88DB58DAC77756548B517E246EA02755006EBF609B86F05BAB4E166503DD65074D7B6C67E1F7E1B744C7674505274547124D8F9965A0E2AB62FB12946E033720C0675981DB83190E001FDB80BE461318E7C7BA020097C3C1984EFC3267217580DA6052E7410E1E71E0C01786AA8166A6B657B64CF17DBB4737B0ED97A09AAFC7E9F9B3B5BC8A08209B6FE0C2BAC43D02E411B804B6CCA8619C5107B0CA9B1E742B37A01F6A803CA51679D91373D8EDAB108C388EBA088B8B1CF22007657EBB453E043D4DC1CE40550537A9A8B4A846B7D6B5806BD6053D075ACA7F7F800EB230A188926FC0747AA8A975171A63C03A70D22186D5146C39AEE5891686319A88BE0C2D7EB41E2A8C36483D5A6B1BAC5C30345A0ED46E283652AB61BAF3B81128D8B6B573784C5BF6546C67F2242D9225222FF1D7B82DE1CB650567361FF542408057C576E0FEB3C7D105B5848742E7BCD1CB617D8253B33271ADCE7B0E613E5402014243F6844D900E8FFEB06E982CEFB0D44FF002BCE671240176BA68314DE61F766AF2F7D4D24271B23FCE13D500DEB9E8B6966AB56E388F2908516584FAAA593156EADC31AE2FB5D5916C522883C11EE3B0B239C5956375BA73A519363D09A50C11A67A663E082D544501356FA5012C926631E4887C67C261B765895771D99A5E437B080D87E3B2027BBD5640C50D06683E3A56017722165C41E4026B2D9BD3A8216D90106C201F74092CF80B57A3524E0A4B3E106B6BF867F9F0FD5248239B08BDEDED5D006ABEFAB37086605FF468025A5B41EEF9FA5FF7F0F8D1B098F72F9412228B612400F8303A52238111C770E72854C044F0BEB0B2013B35822704DC806D0802844FC8947F88957FC76EA8DF647BC740266408A5E28040F66090D221194C9CAA78609B3DD8A4613E00F104466843CC16CD2214D17A490B010DC760E0C5081B273AD92B937CBA3F0D556B13D81A599FC9201E9472D49A207972046AC8C66AC1BA129465C19536B72C856FA89C22913242C7C701B6B4EF4A352255A2604241BBCA97AB9EC89FA01CF1CD22A774895A273C60B54B9529CE05E3428D219E9E66658638415744D12B636B87780150C19729D6E72840A3518493CCCA28E9C420E45A348A37C409C3CDB17E489CA01FABC151509E17610C01C7404DAEB0FAF51F83455B2AAABF377C8C92AF6844DC5E0B5EE79D82513543497ADA2E573720AB0A2A8D1E29097FCCA7DA19F75A169A16CF6761F088D3C4F638260A39E0E0A19E45872C1A13A1C66F8E43D0F8C319A7606BCE4C9CF7EFEA0F6A0EAA00E2CC8858F78489053C843C1A1C4A1839DED73B8FED1E16E8F8283800CC8'
			$DllBinary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
			$DllBinary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
			$DllBinary &= '504700B967E195B65A20FF18B340EF5E211A3614D8EB435AC089CB609B220B918AA60A4800478238013992557CDD9DA5B2085257037E20EA3BC065890CB9443423C50E51C2E30BB7388B8F4DD65F5AAE9147847513AC129561070FDF91A089F2E2916E153DBC2045143BEE91A2D802A82B667F4EB6A476548F106A1789C59696F09A75F8D8D25FC1755730B6E0CD1F1850456695DB2EF3AA2C163910033D029AB1DB6B33144042481D2E5CB6D706E19333B85B0D38366077B933411A0144E92538CB36F6ED14E014160E010B3F34533461D37DD93565036A017410EB2E24F026F906792E000B02302C3869FC03706972828434BC4A3C6B1DBB5BDC449F2809DB58BB281401EB9503C6FF4BD3EB43D3E329EB89D9FA2E90B7EF1082D80C039C2059548B83005FE04A384984F7D121C8C3726F9D31B6031628CC1C215A148D01DDB6F75C1A18896B1703F26C61B919FA2D9274ED7B14EF0C2460669AAD7AE031C9DDD1D10416BF1DB0FF112BC283D900B302B4A093BF7D20EB0F10301B4A34B3044023F8BD0C47DB886858A3D4870101E40D6C0411FA0CF6804A17200D095F200EF73FC88DE8929679058D4B280352542B15EF852604FF39D176178D5AD85C601E861BE4DFB9025FC85D818F4525600B6E5619343C552CCB05C21940310AB8CD41C4F75635AB98D0390AB5F70004B51B40043226C92B458D4D06CCE7F927E658E1B3C986FEED2E1700EB250FB02961360E1407265FD1404213AC4519980262C5744B6649B4FF016B02870E874CC278C744C27CC8EEAF370493878CC288B284C28C06CF8A2E7E1309C96D8FE99BB2650744EB5A5BEBAAF7748EF6176DB60B076C88C191D3E085FAC4786BB62D5E0908EB848B19C3808521040214ECC05507D10572D8C31C010DF133E20F817A384072141E199BEBC6A8FD2B6B485BF5686FD2498D44E08EF129E4632C5A9046A1B12B6F0B446D295DE5F67620740FBB0DB7FD014A1C837A2CDB0663B22C144074902D61951B622030B71782ED300508010D837B807507365CC6F6804B24802B24FE03BAC328558C5233E525FE2168566DD38DF24976040FEE720D6816A35B8D87878A4F3C1072815074F0AEA8090B5614034B0CC2674098138606EB077E02607808732257090C56500E0C2B4204EAB7842287EA80F18C366BEEB094C27E702E701B5EC9324A4A0D2BF7894C7942532DE9372BF6898C64032D1612CFA98F8CC880801B0855A14BF11C48A5B69E386CA3B3805494B015E33197630030FD4CC401592902E0F04A05F6D2C80D4121513474230E726CF65E894165CEC08B398C97B06242792A1D6C131D8CB54260FA3CAC3E5C983B6481FA0832065A15ED80643B35EBB48F36E62379C710D98B149CAB6056227C259BBD64CBFC459FD4B61C72F27D608951647B4E057E654026EC358A68DD86DBDA6C3B38'
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary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
			$DllBinary &= '0001000000300000800000000003FD9F4B000000000000010000000000480000005CC0010030020000000000000000000058900100300234000000560053005F00560045005200530049004F004E005F0049004E0046004F0000000000BD04EFFE00000100440001000000000044000100000000000000000000000000040000000100000000000000000000000000000090010000010053007400720069006E006700460069006C00650049006E0066006F0000006C010000010030003400300039003000340045003400000046000F000100460069006C0065004400650073006300720069007000740069006F006E000000909066006C0061007400200061007300730065006D0062006C0065007200000090907200270001004C006500670061006C0043006F007000790072006900670068007400000043006F0070007900720069006700680074002000A900200031003900390039002D003200300030003900200054006F006D00610073007A00200047007200790073007A007400610072002E00000090902A0005000100460069006C006500560065007200730069006F006E000000909031002E0036003800000090902E0005000100500072006F006400750063007400560065007200730069006F006E00000031002E0036003800000090903A00090001004F0072006900670069006E0061006C00460069006C0065006E0061006D00650000004600410053004D002E0044004C004C0000009090440000000100560061007200460069006C00650049006E0066006F00000000002400040000005400720061006E0073006C006100740069006F006E000000000009040000000000000000000000000000CCC20100B4C201000000000000000000000000000000000000000000D8C20100E6C20100F6C2010006C3010014C30100000000004B45524E454C33322E444C4C00004C6F61644C69627261727941000047657450726F634164647265737300005669727475616C50726F7465637400005669727475616C416C6C6F6300005669727475616C467265650000000000000000000000000000006AC301000100000003000000030000004CC3010058C3010064C3010008300000233000000E30000073C3010081C3010093C301000100020000004641534D2E444C4C006661736D5F417373656D626C65006661736D5F417373656D626C6546696C65006661736D5F47657456657273696F6E000000B001000C0000008D3A00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'

		$_FasmDll = MemoryDllOpen($DllBinary)
		If Not $_FasmDll Then _FasmFailed()
	EndIf

	Local $FasmObj[5] = [DllStructCreate("int state;int length;ptr output;byte binary[" & $MaxStructSize & "]"), "" & @LF, False, 0, $PassesLimit]
	Return $FasmObj
EndFunc

Func FasmExit(ByRef $FasmObj)
	$FasmObj = 0
	$_FasmLastError = 0
	MemoryDllClose($_FasmDll)
	$_FasmDll = 0
EndFunc

Func FasmGetVersion()
	Local $Ret = MemoryDllCall($_FasmDll, "uint", "fasm_GetVersion")
	If Not IsArray($Ret) Then _FasmFailed()
	Return BitAND($Ret[0], 0xffff) & "." & BitShift($Ret[0], 16)
EndFunc

Func _FasmCheck(ByRef $FasmObj)
	If IsArray($FasmObj) And UBound($FasmObj) = 5 And IsDllStruct($FasmObj[0]) Then
		Return True
	EndIf
	Return False
EndFunc

Func _FasmFailed()
	MsgBox(16, "Flat Assembler Failed", "Fasm.dll or MemoryDll is not working!")
	Exit	
EndFunc

Func _FasmError()
	Local $Error = FasmGetLastError()
	If IsArray($Error) And UBound($Error) = 4 Then
		Local $ErrorCode = $Error[0], $ErrorMsg = $Error[1], $ErrorLineNumber = $Error[2], $ErrorLine = $Error[3]
		If $ErrorLine >= 0 Then $ErrorMsg &= @CRLF & "LINE NUMBER: " & $ErrorLineNumber & @CRLF & 'ERROR LINE: "' & $ErrorLine & '"'

		MsgBox(16, "Flat Assembler Error: " & $ErrorCode, $ErrorMsg)
	EndIf
	FasmExit($_FasmLastError)
	Exit
EndFunc

Func FasmGetLastError()
	If _FasmCheck($_FasmLastError) Then
		Local $ErrorCode, $ErrorMsg, $ErrorLineNumber = -1, $ErrorLine

		$ErrorCode = $_FasmLastError[$_foSTATE]
		Switch $_FasmLastError[$_foSTATE]
		Case $FASM_INVALID_PARAMETER
			$ErrorMsg = "INVALID PARAMETER"
		Case $FASM_OUT_OF_MEMORY
			$ErrorMsg = "OUT OF MEMORY"
		Case $FASM_STACK_OVERFLOW
			$ErrorMsg = "STACK OVERFLOW"
		Case $FASM_SOURCE_NOT_FOUND
			$ErrorMsg = "SOURCE NOT FOUND"
		Case $FASM_UNEXPECTED_END_OF_SOURCE
			$ErrorMsg = "UNEXPECTED END OF SOURCE"
		Case $FASM_CANNOT_GENERATE_CODE
			$ErrorMsg = "CANNOT GENERATE CODE"
		Case $FASM_FORMAT_LIMITATIONS_EXCEDDED
			$ErrorMsg = "FORMAT LIMITATIONS EXCEDDED"
		Case $FASM_WRITE_FAILED
			$ErrorMsg = "WRITE FAILED"
		Case $FASM_ERROR
			$ErrorCode = DllStructGetData($_FasmLastError[$_foSTRUCT], "length")
			Local $LineHeader = DllStructCreate("ptr;int;uint;uint", Ptr(DllStructGetData($_FasmLastError[$_foSTRUCT], "output")))

			If DllStructGetData($LineHeader, 2) < 0 Then
				Local $MacroLineHeader = DllStructCreate("ptr;int;uint;uint", Ptr(DllStructGetData($LineHeader, 4)))
				$LineHeader = $MacroLineHeader
			EndIf
			$ErrorLineNumber = DllStructGetData($LineHeader, 2)

			Local $Lines = StringSplit($_FasmLastError[$_foSOURCE], @LF)
			$ErrorLine = $Lines[$ErrorLineNumber]

			Switch DllStructGetData($_FasmLastError[$_foSTRUCT], "length")
			Case $FASMERR_FILE_NOT_FOUND
				$ErrorMsg = "FILE NOT FOUND"
			Case $FASMERR_ERROR_READING_FILE
				$ErrorMsg = "ERROR READING FILE"
			Case $FASMERR_INVALID_FILE_FORMAT
				$ErrorMsg = "INVALID FILE FORMAT"
			Case $FASMERR_INVALID_MACRO_ARGUMENTS
				$ErrorMsg = "INVALID MACRO ARGUMENTS"
			Case $FASMERR_INCOMPLETE_MACRO
				$ErrorMsg = "INCOMPLETE MACRO"
			Case $FASMERR_UNEXPECTED_CHARACTERS
				$ErrorMsg = "UNEXPECTED CHARACTERS"
			Case $FASMERR_INVALID_ARGUMENT
				$ErrorMsg = "INVALID ARGUMENT"
			Case $FASMERR_ILLEGAL_INSTRUCTION
				$ErrorMsg = "ILLEGAL INSTRUCTION"
			Case $FASMERR_INVALID_OPERAND
				$ErrorMsg = "INVALID OPERAND"
			Case $FASMERR_INVALID_OPERAND_SIZE
				$ErrorMsg = "INVALID OPERAND SIZE"
			Case $FASMERR_OPERAND_SIZE_NOT_SPECIFIED
				$ErrorMsg = "OPERAND SIZE NOT SPECIFIED"
			Case $FASMERR_OPERAND_SIZES_DO_NOT_MATCH
				$ErrorMsg = "OPERAND SIZES DO NOT MATCH"
			Case $FASMERR_INVALID_ADDRESS_SIZE
				$ErrorMsg = "INVALID ADDRESS SIZE"
			Case $FASMERR_ADDRESS_SIZES_DO_NOT_AGREE
				$ErrorMsg = "ADDRESS SIZES DO NOT AGREE"
			Case $FASMERR_PREFIX_CONFLICT
				$ErrorMsg = "PREFIX CONFLICT"
			Case $FASMERR_LONG_IMMEDIATE_NOT_ENCODABLE
				$ErrorMsg = "LONG IMMEDIATE NOT ENCODABLE"
			Case $FASMERR_RELATIVE_JUMP_OUT_OF_RANGE
				$ErrorMsg = "RELATIVE JUMP OUT OF RANGE"
			Case $FASMERR_INVALID_EXPRESSION
				$ErrorMsg = "INVALID EXPRESSION"
			Case $FASMERR_INVALID_ADDRESS
				$ErrorMsg = "INVALID ADDRESS"
			Case $FASMERR_INVALID_VALUE
				$ErrorMsg = "INVALID VALUE"
			Case $FASMERR_VALUE_OUT_OF_RANGE
				$ErrorMsg = "VALUE OUT OF RANGE"
			Case $FASMERR_UNDEFINED_SYMBOL
				$ErrorMsg = "UNDEFINED SYMBOL"
			Case $FASMERR_INVALID_USE_OF_SYMBOL
				$ErrorMsg = "INVALID USE OF SYMBOL"
			Case $FASMERR_NAME_TOO_LONG
				$ErrorMsg = "NAME TOO LONG"
			Case $FASMERR_INVALID_NAME
				$ErrorMsg = "INVALID NAME"
			Case $FASMERR_RESERVED_WORD_USED_AS_SYMBOL
				$ErrorMsg = "RESERVED WORD USED AS SYMBOL"
			Case $FASMERR_SYMBOL_ALREADY_DEFINED
				$ErrorMsg = "SYMBOL ALREADY DEFINED"
			Case $FASMERR_MISSING_END_QUOTE
				$ErrorMsg = "MISSING END QUOTE"
			Case $FASMERR_MISSING_END_DIRECTIVE
				$ErrorMsg = "MISSING END DIRECTIVE"
			Case $FASMERR_UNEXPECTED_INSTRUCTION
				$ErrorMsg = "UNEXPECTED INSTRUCTION"
			Case $FASMERR_EXTRA_CHARACTERS_ON_LINE
				$ErrorMsg = "EXTRA CHARACTERS ON LINE"
			Case $FASMERR_SECTION_NOT_ALIGNED_ENOUGH
				$ErrorMsg = "SECTION NOT ALIGNED ENOUGH"
			Case $FASMERR_SETTING_ALREADY_SPECIFIED
				$ErrorMsg = "SETTING ALREADY SPECIFIED"
			Case $FASMERR_DATA_ALREADY_DEFINED
				$ErrorMsg = "DATA ALREADY DEFINED"
			Case $FASMERR_TOO_MANY_REPEATS
				$ErrorMsg = "TOO MANY REPEATS"
			EndSwitch
		EndSwitch

		Local $Error[4] = [$ErrorCode, $ErrorMsg, $ErrorLineNumber, $ErrorLine]
		Return $Error
	Else
		Local $Error[4] = [0, "", -1, ""]
		Return $Error
	EndIf
EndFunc

Func FasmCompile(ByRef $FasmObj)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER, False)

	If $FasmObj[$_foCOMPILED] Then
		Return SetError($FasmObj[$_foSTATE] <> $FASM_OK, $FasmObj[$_foSTATE], $FasmObj[$_foSTATE] <> $FASM_OK)
	Else
		Local $StructPtr = DllStructGetPtr($FasmObj[$_foSTRUCT])
		Local $StructSize = DllStructGetSize($FasmObj[$_foSTRUCT])
		Local $Ret = MemoryDllCall($_FasmDll, "int", "fasm_Assemble", "str", $FasmObj[$_foSOURCE], "ptr", $StructPtr, "uint", $StructSize, "uint", $FasmObj[$_foPASSLMT], "uint", 0)
		If Not IsArray($Ret) Then _FasmFailed()

		$FasmObj[$_foCOMPILED] = TRUE
		$FasmObj[$_foSTATE] = $Ret[0]
		If $FasmObj[$_foSTATE] = $FASM_OK Then
			Local $CodePtr = DllStructGetData($FasmObj[$_foSTRUCT], "output")
			Local $CodeLen = DllStructGetData($FasmObj[$_foSTRUCT], "length")
			Local $Offset = Number($CodePtr - $StructPtr) - 12
			Local $Binary = BinaryMid(DllStructGetData($FasmObj[$_foSTRUCT], "binary"), $Offset + 1, $CodeLen)
			DllStructSetData($FasmObj[$_foSTRUCT], "binary", $Binary)
			Return SetError(0, $FasmObj[$_foSTATE], True)
		Else
			Return SetError(1, $FasmObj[$_foSTATE], False)
		EndIf
	EndIf
EndFunc

Func FasmGetBinary(ByRef $FasmObj)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER, Binary(""))

	If Not $FasmObj[$_foCOMPILED] Then FasmCompile($FasmObj)
	If $FasmObj[$_foSTATE] = $FASM_OK Then
		Local $CodeLen = DllStructGetData($FasmObj[$_foSTRUCT], "length")
		Return BinaryMid(DllStructGetData($FasmObj[$_foSTRUCT], "binary"), 1, $CodeLen)
	Else
		$_FasmLastError = $FasmObj
		Return SetError(1, $FasmObj[$_foSTATE], Binary(""))
	EndIf
EndFunc

Func FasmGetFuncPtr(ByRef $FasmObj)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER, Ptr(0))

	If Not $FasmObj[$_foCOMPILED] Then FasmCompile($FasmObj)
	If $FasmObj[$_foSTATE] = $FASM_OK Then
		Return DllStructGetPtr($FasmObj[$_foSTRUCT], "binary")
	Else
		$_FasmLastError = $FasmObj
		Return DllCallbackGetPtr(DllCallbackRegister("_FasmError", "int:cdecl", ""))
	EndIf
EndFunc

Func FasmGetBasePtr(ByRef $FasmObj)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER, Ptr(0))

	Return DllStructGetPtr($FasmObj[$_foSTRUCT], "binary")
EndFunc

Func FasmAdd(ByRef $FasmObj, $Text)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER)

	If $Text Then
		$FasmObj[$_foSOURCE] &= $Text & @LF
		$FasmObj[$_foCOMPILED] = False
	EndIf
EndFunc

Func FasmGetSource(ByRef $FasmObj)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER, "")

	Return $FasmObj[$_foSOURCE]
EndFunc

Func FasmSetSource(ByRef $FasmObj, $Text)
	If Not _FasmCheck($FasmObj) Then Return SetError(1, $FASM_INVALID_PARAMETER)

	$FasmObj[$_foSOURCE] = $Text
	$FasmObj[$_foCOMPILED] = False
EndFunc

Func FasmReset(ByRef $FasmObj)
	FasmSetSource($FasmObj, "")
EndFunc
